<?php

/**
 * Créé un nouveau noeud clôné issu d'une requête ADT :
 * 
 * @param $type Le type de contenu associé à la requête
 * @param $id L'identifiant de la fiche
 * @param $title Le titre de la fiche
 * @param $rawData Les données XML brutes (string) de la fiche
 */
function createADTNode($activite, $type, $id, $title, $rawData)
{
    //-- création du noeud
    $node = new stdClass();
    $node->type = $type;
    node_object_prepare($node);
    
    //-- champs de base
    $node->title = $title;
    $node->language = LANGUAGE_NONE;    
    $node->{variable_get('FIELD_ADT_NODE_ID')}['und'][0]['value'] = $id;
    $node->{variable_get('FIELD_ADT_NODE_XML_DATA')}['und'][0]['value'] = $rawData;
    $node->path = array('alias' => buildCloneAlias($activite, $title), 'pathauto' => FALSE);
    
    //-- parse la chaîne en XML
    $xml = @simplexml_load_string($rawData);
    //$xml = $xmlRoot->obj;
    
    //-- remplit les champs "prédéfinis" (ex: km, duree, descriptifs...)
    fillDefaultFields($node, $xml, true);
    
    //-- sauvegarde le noeud
    node_save($node);
}

/**
 * Construit l'alias pour un noeud ADT
 * 
 * @param $activite L'activité à laquelle le noeud est rattaché
 * @param $title Le titre "brut" du noeud
 */
function buildCloneAlias($activite, $title)
{    
     require_once('sites/all/libraries/Inflector.php');
     
     $alias = strtolower(strip_tags($title));
        
     $inflector = new Inflector();
     $alias = $inflector->slug($alias, '-');
     
     $parent_path = join('/', explode('|', $activite->parent_path));
     $alias = $parent_path . '/' . $alias;
     
     return $alias;
}



/**
 * Met à jour un noeud clôné :
 * 
 * @param $nid L'identifiant du noeud à mettre à jour
 * @param $rawData Les nouvelles données XML à appliquer au noeud
 */
function updateADTNode($nid, $rawData)
{
    $node = node_load($nid);
    $node->{variable_get('FIELD_ADT_NODE_XML_DATA')}['und'][0]['value'] = $rawData;
   
    $xml = @simplexml_load_string($rawData);
    
    fillDefaultFields($node, $xml);
    
    $title = $node->title;
    $activite = getActiviteByCloneType($node->type);
    
    $node->path = array('alias' => buildCloneAlias($activite, $title), 'pathauto' => FALSE);
    if ($node->type == 'clone_adt_escalade') dpm($node);
    node_save($node);
}

/**
 * Remplit les champs prédéfinis dans le type de contenu :
 *      -> extrait certaines valeurs du XML pour les stocker dans des champs
 *      -> prend en compte uniquement les champs de texte simple
 *      -> les champs de texte long sont gérés manuellement sur la page d'édition du noeud (AJAX)
 * 
 * @param type $node
 * @param type $xml
 */
function fillDefaultFields($node, $xml, $syncLongTexts = false)
{
    //-- inclut la librairie de fonctions utilitaires
    require_once('sites/all/libraries/utils.php');
    
    //-- recherche les champs du type de contenu
    $fields = field_info_instances("node", $node->type);
    
    foreach($fields as $field)
    {
        //-- ignore le champ id et le champ xmlData (déja traités)
        if ($field['field_name'] == variable_get('FIELD_ADT_NODE_ID') || $field['field_name'] == variable_get('FIELD_ADT_NODE_XML_DATA'))
            continue;
        
        //-- on ne traite pas les champs de texte long (synchro manuelle)
        $isTextArea = $field['widget']['type'] == 'text_textarea';
        
        if (!$isTextArea || $syncLongTexts)
        {        
            //-- cherche la valeur par défaut du champ
            $defaultValue = $field['default_value']['0']['value'];

            //-- la valeur par défaut indique le noeud XML à extraire
            if ($defaultValue != NULL)
            {
                $value = @extractFieldValue($xml, $defaultValue);
                
                $node->{$field['field_name']}['und'][0]['value'] = nl2br2($value);
            }
        }
    }
}

/**
 *  Fonction appelée en asynchrone pour synchroniser un noeud :
 *      -> appelée automatiquement en JS lors de l'affichage d'un noeud
 */
function ajax_sync_node_callback($nid, $requestNid)
{
    //-- charge le noeud
    $node = node_load($nid);
    $id = $node->{variable_get('FIELD_ADT_NODE_ID')}['und'][0]['value'];
    
    //-- charge la requête associée à la fiche
    $adtRequest = node_load($requestNid);    
    $table = $adtRequest->{variable_get('FIELD_ADT_TABLE')}['und'][0]['value'];
    $support = $adtRequest->{variable_get('FIELD_ADT_SUPPORT')}['und'][0]['value'];
    $params = $adtRequest->{variable_get('FIELD_ADT_PARAMS')}['und'][0]['value'];
    
    //-- lance la requête ADT
    $rawData = sendDataXMLRequest($table, $support, $params, $id);
    
    //-- met à jour le noeud
    $node->{variable_get('FIELD_ADT_NODE_XML_DATA')}['und'][0]['value'] = $rawData;
   
    //-- remplit les champs par défaut
    $xml = @simplexml_load_string($rawData);
    fillDefaultFields($node, $xml);
    
    //-- sauvegarde le noeud
    node_save($node);
    
    echo 'success';
}


/**
 *  Fonction appelée en asynchrone pour synchroniser un champ particulier :
 *      -> appelée depuis la page d'édition du noeud
 */
function ajax_sync_field_callback()
{
    if (!isset($_POST['nid']) || !isset($_POST['fieldName']))
        die('Paramètres manquants');
    
    //-- extraction des paramètres
    $nid = $_POST['nid'];
    $fieldName = $_POST['fieldName'];
    $node = node_load($nid);
    $nodeType = $node->type;
    $idFiche = $node->{variable_get('FIELD_ADT_NODE_ID')}['und'][0]['value'];
    
    //-- recherche l'activité liée
    $activite = getActiviteByCloneType($nodeType);
    
    //-- si aucune activité n'est trouvée, renvoie une erreur
    if (!$activite){
        die('Aucune activité ne correspond au noeud en cours.');
    }
    
    //-- charge le noeud ADT_Request correspondant à l'activité
    $requete = node_load($activite->adt_request_nid);
    
    //-- envoie la requête au serveur
    $xml = executeADTRequest($requete, $idFiche);
    
    //-- recherche les champs du type de contenu
    $fields = field_info_instances("node", $nodeType);
    
    foreach($fields as $field)
    {
        //-- on ne traite que le champ qui nous intéresse
        if ($field['field_name'] != $fieldName)
            continue;
        
        //-- inclut la librairie de fonctions utilitaires
        require_once('sites/all/libraries/utils.php');

        //-- recherche la valeur par défaut du champ
        $path = $field['default_value']['0']['value'];
        
        //-- extrait la valeur correspondante dans le XML
        $value = extractFieldValue($xml, $path);
        
        //-- construit la réponse à renvoyer au JS
        $response = array(
            'data' => nl2br2($value),
        );
        
        echo json_encode($response);
        return;
    }
    
    die('Aucun résultat n\'a été trouvé');
}

/**
 *  Lancer une recherche xpath dans le XML pour trouver une valeur particulière :
 */
function extractFieldValue($xml, $path)
{
    $data = $xml->obj->xpath($path);
    
    if (!$data)
    {
        dpm('Champ introuvable : ' . $path . ' ('. $xml->obj->societe . ')');
        //dpm($xml);
        return '';
    }
    else
    {
        return (string) $data[0][0];
    }
}

?>
